/****************************************************************************/
/**                                                                        **/
/**                               RDBitmap.c                               **/
/**                                                                        **/
/** Routines to read in Windows bitmap format graphics files               **/
/**                                                                        **/
/** Copyright (C) Marcel de Kogel 1997                                     **/
/**     You are not allowed to distribute this software commercially       **/
/**     Please, notify me, if you make any changes to this file            **/
/****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "RDBitmap.h"

typedef unsigned char byte;

static short bytes2short (byte *p)
{
 return p[0]+(p[1]<<8);
}

static int bytes2int (byte *p)
{
 return p[0]+(p[1]<<8)+(p[2]<<16)+(p[3]<<24);
}

/****************************************************************************/
/* Return values:                                                           */
/* >=0 - Succes                                                             */
/* -1  - File open error                                                    */
/* -2  - Read error                                                         */
/* -3  - File type error                                                    */
/* -4  - Incompatible file type                                             */
/* -5  - Out of memory                                                      */
/****************************************************************************/
int ReadBitmap (char *szFileName,int *nWidth,int *nHeight,
                unsigned char **pBitmapBytes)
{
 int i,j,k,nBytesPerRow,nBitsPerPixel;
 FILE *f;
 byte bitmapheader[54],palette[768],*rowbuf,*p;
 /* Open file */
 f=fopen (szFileName,"rb");
 if (!f) return -1;
 /* Read the header */
 if (fread(bitmapheader,1,54,f)!=54)
 {
  fclose (f);
  return -2;
 }
 /* Check file type */
 if (bitmapheader[0]!='B' || bitmapheader[1]!='M')
 {
  fclose (f);
  return -3;
 }
 /* Check number of planes */
 if (bytes2short(bitmapheader+26)!=1)
 {
  fclose (f);
  return -4;
 }
 /* Check compression method */
 if (bytes2int(bitmapheader+30))
 {
  fclose (f);
  return -4;
 }
 *nWidth=bytes2int(bitmapheader+18);
 *nHeight=bytes2int(bitmapheader+22);
 nBitsPerPixel=bytes2short(bitmapheader+28);
 /* Check for valid width, height, bits per pixel */
 if (nWidth<=0 || nHeight<=0 ||
     (nBitsPerPixel!=1 && nBitsPerPixel!=4 && nBitsPerPixel!=8 &&
      nBitsPerPixel!=24))
 {
  fclose (f);
  return -4;
 }
 /* Read in palette */
 if (nBitsPerPixel<=8)
 {
  if (fseek(f,bytes2int(bitmapheader+14)+14,SEEK_SET))
  {
   fclose (f);
   return -2;
  }
  memset (palette,0,768);
  j=bytes2int (bitmapheader+46);
  if (j>256) j=256;
  if (j==0) j=1<<nBitsPerPixel;
  for (i=0;i<j;++i)
  {
   if ((k=fgetc(f))==EOF)
   {
    fclose (f);
    return -2;
   }
   palette[i*3+2]=k;
   if ((k=fgetc(f))==EOF)
   {
    fclose (f);
    return -2;
   }
   palette[i*3+1]=k;
   if ((k=fgetc(f))==EOF)
   {
    fclose (f);
    return -2;
   }
   palette[i*3+0]=k;
   if (fgetc(f)==EOF)
   {
    fclose (f);
    return -2;
   }
  }
 }
 /* Seek to bitmap data */
 if (fseek(f,bytes2int(bitmapheader+10),SEEK_SET))
 {
  fclose (f);
  return -2;
 }
 /* Get number of bytes per row in bitmap file and allocate row buffer */
 nBytesPerRow=(*nWidth)*nBitsPerPixel/8;
 if (nBytesPerRow&3) nBytesPerRow=(nBytesPerRow&(~3))+4;
 rowbuf=malloc (nBytesPerRow);
 if (!rowbuf)
 {
  fclose (f);
  return -5;
 }
 /* Allocate bitmap buffer */
 *pBitmapBytes=malloc ((*nWidth)*(*nHeight)*3);
 if (!*pBitmapBytes)
 {
  fclose (f);
  return -5;
 }
 /* Read in file */
 p=*pBitmapBytes+(*nWidth)*(*nHeight-1)*3;
 for (i=0;i<*nHeight;++i,p-=2*(*nWidth)*3)
 {
  if (fread(rowbuf,1,nBytesPerRow,f)!=nBytesPerRow)
  {
   fclose (f);
   free (rowbuf);
   free (pBitmapBytes);
   return -2;
  }
  for (j=0;j<*nWidth;++j,p+=3)
   switch (nBitsPerPixel)
   {
    case 24: p[0]=rowbuf[j*3+0];
             p[1]=rowbuf[j*3+1];
             p[2]=rowbuf[j*3+2];
             break;
    case 8:  k=rowbuf[j];
             p[2]=palette[k*3+0];
             p[1]=palette[k*3+1];
             p[0]=palette[k*3+2];
             break;
    case 4:  k=rowbuf[j/2]; if (j&1) k&=15; else k>>=4;
             p[2]=palette[k*3+0];
             p[1]=palette[k*3+1];
             p[0]=palette[k*3+2];
             break;
    case 1:  k=(rowbuf[j/8]>>(7-(j&7)))&1;
             p[2]=palette[k*3+0];
             p[1]=palette[k*3+1];
             p[0]=palette[k*3+2];
             break;
   }
 }
 free (rowbuf);
 fclose (f);
 return 0;
}
