/** ADAMEm: Coleco ADAM emulator ********************************************/
/**                                                                        **/
/**                               bmp2pp.c                                 **/
/**                                                                        **/
/** This program converts Windows bitmap files to PowerPaint format        **/
/**                                                                        **/
/** Copyright (C) Marcel de Kogel 1997                                     **/
/**     You are not allowed to distribute this software commercially       **/
/**     Please, notify me, if you make any changes to this file            **/
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#ifdef MSDOS
#include <sys/stat.h>
#endif
#include "RDBitmap.h"
#include "Bitmap.h"

#ifdef GOT_INLINE
#define INLINE inline
#else
#define INLINE
#endif

typedef unsigned char byte;

#define MAXERR  32

static int debug=0;
static int bdif[256],gdif[256],rdif[256];

/****************************************************************************/
/* Get offset into PowerPaint file buffer                                   */
/****************************************************************************/
static INLINE int get_pp_offset (int width,int height)
{
 int celnr=0;
 if (width>=240) { celnr=1; width-=240; }
 celnr+=(height/160)*2;
 height%=160;
 return 16 + (width/8)*8 + (height/8)*256 + height%8 + celnr*10240;
}

/* colour difference */
#define C_DIF(r,g,b,r2,g2,b2)  ( rdif[abs(r-r2)]+ \
                                 gdif[abs(g-g2)]+ \
                                 bdif[abs(b-b2)] )
/****************************************************************************/
/* Get nearest colour of given colour in given palette                      */
/****************************************************************************/
static int get_nearest_colour (int r,int g,int b,byte *palette)
{
 int i,j,k,l;
 i=1000000000; j=0;
 for (k=1;k<16;++k)
 {
  l=C_DIF (r,g,b,palette[k*3+0],palette[k*3+1],palette[k*3+2]);
  if (l<i)
  {
   j=k; i=l;
   if (!i) return j;
  }
 }
 if (!j) j=1;
 return j;
}

/****************************************************************************/
/* Interpolate 4x4 box. Called three times per pixel                        */
/****************************************************************************/
static INLINE byte interpolate_colour
        (byte *p,int width,unsigned dx,unsigned dy)
{
 unsigned a,b;
 a=(p[3]*dx+p[0]*(65536-dx))>>8;
 b=(p[(width+1)*3]*dx+p[width*3]*(65536-dx))>>8;
 return (b*dy+a*(65536-dy))>>24;
}

/****************************************************************************/
/* Scale given bitmap to given dimensions                                   */
/****************************************************************************/
static byte *ScaleBitmap
 (byte *old_bmp,int old_width,int old_height,
  int new_width,int new_height,int interpolate)
{
 byte *new_bmp,*p,*q;
 unsigned i,j,xp,yp,dx,dy,x,y;
 new_bmp=malloc (new_width*new_height*3);
 if (!new_bmp) return NULL;
 xp=old_width*65536/new_width;
 yp=old_height*65536/new_height;
 p=new_bmp;
 if (interpolate)
 {
  for (i=0,dy=0;i<new_height;++i,dy+=yp)
  {
   for (j=0,dx=0;j<new_width;++j,dx+=xp,p+=3)
   {
    y=dy>>16; x=dx>>16;
    /* Check if we're on one or more edges. If so, do not interpolate */
    if (y>=old_height-1 || x>=old_width-1)
    {
     y+=(dy>>15)&1; if (y>=old_height) y=old_height-1;
     x+=(dx>>15)&1; if (x>=old_width) x=old_width-1;
     q=old_bmp+(y*old_width+x)*3;
     p[0]=q[0];
     p[1]=q[1];
     p[2]=q[2];
    }
    else
    {
     q=old_bmp+((dy>>16)*old_width+(dx>>16))*3;
     y=dy&65535; x=dx&65535;
     p[0]=interpolate_colour(q+0,old_width,x,y);
     p[1]=interpolate_colour(q+1,old_width,x,y);
     p[2]=interpolate_colour(q+2,old_width,x,y);
    }
   }
  }
 }
 else
 {
  for (i=0,dy=0;i<new_height;++i,dy+=yp)
  {
   y=(dy>>16)+((dy>>15)&1); if (y>=old_height) y=old_height-1;
   for (j=0,dx=0;j<new_width;++j,dx+=xp,p+=3)
   {
    x=(dx>>16)+((dx>>15)&1); if (x>=old_width) x=old_width-1;
    q=old_bmp+(y*old_width+x)*3;
    p[0]=q[0];
    p[1]=q[1];
    p[2]=q[2];
   }
  }
 }
 return new_bmp;
}

/****************************************************************************/
/* Apply given filter to bitmap                                             */
/****************************************************************************/
static byte *FilterBitmap (byte *bmp,int width,int height,int *filter)
{
 int i,x,y,b;
 byte *p,*q,*new_bmp;
 new_bmp=malloc (width*height*3);               /* Allocate memory          */
 if (!new_bmp) return NULL;
 b=width*3;
 memcpy (new_bmp,bmp,b);                        /* Top row                  */
 p=new_bmp+b+3;
 q=bmp+b+3;
 for (y=1;y<height-1;++y,p+=6,q+=6)
 {
  p[-3]=q[-3]; p[-2]=q[-2]; p[-1]=q[-1];        /* Left column              */
  for (x=3;x<b-3;++x,++p,++q)                   /* Interior                 */
  {
   /* Weight all 9 pixels according to filter settings */
   i=q[-b-3]*filter[2]+q[-b]*filter[3]+q[-b+3]*filter[4]+
     q[-3]*filter[5]+q[0]*filter[6]+q[3]*filter[7]+
     q[b-3]*filter[8]+q[b]*filter[9]+q[b+3]*filter[10];
   /* Divide by division factor */
   i=i/filter[0];
   /* Add bias */
   i+=filter[1];
   /* Check edges */
   if (i<0) i=0;
   if (i>255) i=255;
   /* Store newly calculated colour value */
   p[0]=i;
  }
  p[0]=q[0]; p[1]=q[1]; p[2]=q[2];              /* Right column             */
 }
 memcpy (p-3,q-3,b);                            /* Bottom row               */
 return new_bmp;
}

/****************************************************************************/
/* Convert given file                                                       */
/****************************************************************************/
static int bmp2pp
  (char *filename,byte *palette,int dither,
   int w1,int w2,int w3,int w4,int noiselevel,
   int scale,int interpolate,int centre,
   float _gamma,int brightness,int greyscale,int *filter)
{
 FILE *f;
 char out[256],gamma_table[256];
 byte *p,*q,*bmp,colbuf[8];
 int *err1,*err2,*errtemp;
 static byte pp[81920];
 static int errbuf1[482*3],errbuf2[482*3],coldif[16][16];
 int width,height,c,i,j,k,l,m,n,o,r,g,b,x,start_x,start_y,n1,n2;
 /* Check for valid noise level */
 if (noiselevel<0) noiselevel=0;
 if (noiselevel>100) noiselevel=100;
 /* Check for valid brightness level */
 if (brightness<0) brightness=0;
 if (brightness>100) brightness=100;
 brightness=(brightness<<1)-100;
 /* Calculate dither weights */
 i=w1+w2+w3+w4;
 if (!i)
 {
  w1=7*4096;
  w2=5*4096;
  w3=3*4096;
  w1=1*4096;
 }
 else
 {
  w1=w1*65536/i;
  w2=w2*65536/i;
  w3=w3*65536/i;
  w4=w4*65536/i;
 }
 /* If output file is given, use it. Else, use the input file with extension
    ".pp" */
 p=strchr(filename,',');
 if (p)
 {
  strcpy (out,p+1);
  *p='\0';
 }
 else
 {
#ifdef MSDOS
/* If running on an MS-DOS machine, convert all backslashes to forward ones */
  _fixpath (filename,out);
#else
  strcpy (out,filename);
#endif
  p=out;
  q=strchr(p,'/');
  while (q)                             /* get last '/'                     */
  {
   p=++q;
   q=strchr(q,'/');
  };
  q=NULL;
  while ((p=strchr(p,'.'))!=NULL)       /* get last '.'                     */
  {
   q=p;
   ++p;
  }
  if (q) *q='\0';                       /* remove extension                 */
  strcat (out,".pp");                   /* add ".pp"                        */
 }
 printf ("Converting %s to %s...\n",filename,out);
 printf (" Loading bitmap file... ");
 fflush (stdout);
 i=ReadBitmap (filename,&width,&height,&bmp);
 if (i<0)
 {
  printf ("FAILED: ");
  switch (i)
  {
   case -1: puts ("Can't open file");
            break;
   case -2: puts ("Read error");
            break;
   case -3: puts ("File is not Windows bitmap graphics file");
            break;
   case -4: puts ("Incompatible file type");
            break;
   case -5: puts ("Out of memory");
            break;
   default: puts ("Unkown error");
            break;
  }
  return i;
 }
 p=bmp;
 if (_gamma<=0.01) _gamma=0.01;
 if (_gamma!=1.0)
 {
  printf ("OK\n Gamma-adjusting... ");
  fflush (stdout);
  for (i=0;i<256;++i)
  {
   j=0.5+255*pow(i/255.0,1.0/_gamma);
   if (j>255) j=255;
   gamma_table[i]=j;
  }
  for (i=0;i<width*height*3;i++)
   p[i+0]=gamma_table[p[i+0]];
  if (debug) WriteBitmap ("GAMMA.bmp",24,0,width,width,height,bmp,NULL);
 }
 if (brightness)
 {
  printf ("OK\n %s... ",(brightness<0)? "Darkening":"Brightening");
  fflush (stdout);
  for (i=0;i<width*height*3;i+=3)
  {
   j=p[i+0]+brightness; if (j<0) j=0; if (j>255) j=255;
   p[i+0]=j;
   j=p[i+1]+brightness; if (j<0) j=0; if (j>255) j=255;
   p[i+1]=j;
   j=p[i+2]+brightness; if (j<0) j=0; if (j>255) j=255;
   p[i+2]=j;
  }
  if (debug) WriteBitmap ("BRIGHT.bmp",24,0,width,width,height,bmp,NULL);
 }
 if (greyscale)
 {
  printf ("OK\n Converting to greyscale... ");
  fflush (stdout);
  for (i=0;i<width*height*3;i+=3)
   p[i+0]=p[i+1]=p[i+2]=(p[i+0]*11+p[i+1]*59+p[i+2]*30)/100;
  if (debug) WriteBitmap ("GREY.bmp",24,0,width,width,height,bmp,NULL);
 }
 if (scale)
 {
  int xmax,ymax;
  switch (scale)
  {
   case 2:  xmax=480;
            ymax=320;
            break;
   case 3:  xmax=480;
            ymax=640;
            break;
   default: xmax=240;
            ymax=160;
            break;
  }
  if (height*xmax/width>ymax)
  {
   i=width*ymax/height;
   j=ymax;
  }
  else
  {
   i=xmax;
   j=height*xmax/width;
  }
  printf ("OK\n Scaling to %ux%u... ",i,j);
  fflush (stdout);
  p=ScaleBitmap (bmp,width,height,i,j,interpolate);
  if (!p)
  {
   puts ("FAILED: Out of memory");
   free (bmp);
   return -5;
  }
  free (bmp);
  bmp=p;
  width=i;
  height=j;
  if (debug) WriteBitmap ("SCALE.bmp",24,0,width,width,height,bmp,NULL);
 }
 if (filter)
 {
  printf ("OK\n Filtering... ");
  fflush (stdout);
  p=FilterBitmap (bmp,width,height,filter);
  if (!p)
  {
   puts ("FAILED: Out of memory");
   free (bmp);
   return -5;
  }
  free (bmp);
  bmp=p;
  if (debug) WriteBitmap ("FILTER.bmp",24,0,width,width,height,bmp,NULL);
 }
 if (centre)
 {
  switch (centre)
  {
   case 2:  start_x=480;
            start_y=320;
            break;
   case 3:  start_x=480;
            start_y=640;
            break;
   default: start_x=240;
            start_y=160;
            break;
  }
  if (width>start_x)
   start_x=0;
  else
   start_x=(start_x-width)/2;
  if (height>start_y)
   start_y=0;
  else
   start_y=(start_y-height)/2;
 }
 else
  start_x=start_y=0;
 printf ("OK\n Converting bitmap data...   0%%");
 fflush (stdout);
 /* Clear error buffers */
 memset (errbuf1,0,sizeof(errbuf1));
 memset (errbuf2,0,sizeof(errbuf2));
 err1=errbuf1;
 err2=errbuf2;
 /* Initialise colour difference table */
 for (i=0;i<16;++i)
  for (j=0;j<16;++j)
   coldif[i][j]=C_DIF(palette[i*3+0],palette[i*3+1],palette[i*3+2],
                      palette[j*3+0],palette[j*3+1],palette[j*3+2]);
 /* Initialise PowerPaint structure: Foreground colour=15 (white),
    background colour=1 (black), pattern=0x00 (all background) */
 for (i=0;i<81920;i+=10240)
 {
  memset (pp+i,0xF1,5120);
  memset (pp+i+5120,0x00,5120);
 }
 if (height+start_y>640) height=640-start_y;
 for (i=start_y;i<start_y+height;++i)
 {
  for (j=start_x&(~7);j<480 && j<start_x+width;j+=8)
  {
   /* Convert 8 pixels to ADAM palette */
   for (k=0,x=j;k<8;++k,++x)
   {
    /* Get colour */
    if (x>=start_x && x<start_x+width)
    {
     b=bmp[((i-start_y)*width+x-start_x)*3+0];
     g=bmp[((i-start_y)*width+x-start_x)*3+1];
     r=bmp[((i-start_y)*width+x-start_x)*3+2];
    }
    else b=g=r=0;
    /* Dither */
    if (dither&1)
    {
     n1=(rand()&4095)*noiselevel/100;
     n2=(rand()&4095)*noiselevel/20;
     r+=((w1-n2)*err1[x*3+0]+(w4+n1)*err2[x*3+0]+
         (w2+n2)*err2[x*3+3]+(w3-n1)*err2[x*3+6])>>16;
     g+=((w1-n2)*err1[x*3+1]+(w4+n1)*err2[x*3+1]+
         (w2+n2)*err2[x*3+4]+(w3-n1)*err2[x*3+7])>>16;
     b+=((w1-n2)*err1[x*3+2]+(w4+n1)*err2[x*3+2]+
         (w2+n2)*err2[x*3+5]+(w3-n1)*err2[x*3+8])>>16;
     if (r<0) r=0;  if (r>255) r=255;
     if (g<0) g=0;  if (g>255) g=255;
     if (b<0) b=0;  if (b>255) b=255;
    }
    /* Get and store nearest colour */
    c=get_nearest_colour(r,g,b,palette);
    colbuf[k]=c;
    /* Update error tables */
    if (dither&1)
    {
     r-=palette[c*3+0];
     g-=palette[c*3+1];
     b-=palette[c*3+2];
     if (r<-MAXERR || r>MAXERR) r=(r*3)>>2;
     if (g<-MAXERR || g>MAXERR) g=(g*3)>>2;
     if (b<-MAXERR || b>MAXERR) b=(b*3)>>2;
     err1[x*3+3]=r;
     err1[x*3+4]=g;
     err1[x*3+5]=b;
    }
   }
   /* Get best 2 colours for this character row */
   c=0; l=1000000000;
   for (n=1;n<16;++n)
    for (o=n+1;o<16;++o)
    {
     m=0;
     for (k=0;k<8;++k)
     {
      r=coldif[colbuf[k]][n];
      g=coldif[colbuf[k]][o];
      if (r<g) m+=r; else m+=g;
     }
     if (m<l) { l=m; c=(n<<8)|o; if (!l) break; }
    }
   l=c>>8; m=c&255;
   /* If only one was found, just pick another one */
   if (l==m) l=(~m)&15;
   /* Store colours in colour table */
   pp[get_pp_offset(j,i)]=(l<<4)|m;
   /* Get pattern for this character box */
   for (k=0,x=j,n=0;k<8;++k,++x)
   {
    /* Dither again */
    if (dither&2)
    {
     if (x>=start_x && x<start_x+width)
     {
      b=bmp[((i-start_y)*width+x-start_x)*3+0];
      g=bmp[((i-start_y)*width+x-start_x)*3+1];
      r=bmp[((i-start_y)*width+x-start_x)*3+2];
     }
     else b=g=r=0;
     n1=(rand()&4095)*noiselevel/100;
     n2=(rand()&4095)*noiselevel/20;
     r+=((w1-n2)*err1[x*3+0]+(w4+n1)*err2[x*3+0]+
         (w2+n2)*err2[x*3+3]+(w3-n1)*err2[x*3+6])>>16;
     g+=((w1-n2)*err1[x*3+1]+(w4+n1)*err2[x*3+1]+
         (w2+n2)*err2[x*3+4]+(w3-n1)*err2[x*3+7])>>16;
     b+=((w1-n2)*err1[x*3+2]+(w4+n1)*err2[x*3+2]+
         (w2+n2)*err2[x*3+5]+(w3-n1)*err2[x*3+8])>>16;
     if (r<0) r=0;  if (r>255) r=255;
     if (g<0) g=0;  if (g>255) g=255;
     if (b<0) b=0;  if (b>255) b=255;
     c=m;
     if (C_DIF(palette[l*3+0],palette[l*3+1],palette[l*3+2],r,g,b)<
         C_DIF(palette[m*3+0],palette[m*3+1],palette[m*3+2],r,g,b))
     { n|=1<<(7-k); c=l; }
     r-=palette[c*3+0];
     g-=palette[c*3+1];
     b-=palette[c*3+2];
     if (r<-MAXERR || r>MAXERR) r=(r*3)>>2;
     if (g<-MAXERR || g>MAXERR) g=(g*3)>>2;
     if (b<-MAXERR || b>MAXERR) b=(b*3)>>2;
     err1[x*3+3]=r;
     err1[x*3+4]=g;
     err1[x*3+5]=b;
    }
    else
    {
     /* No dither, just get nearest colour */
     c=colbuf[k];
     r=palette[c*3+0];
     g=palette[c*3+1];
     b=palette[c*3+2];
     if (C_DIF(palette[l*3+0],palette[l*3+1],palette[l*3+2],r,g,b)<
         C_DIF(palette[m*3+0],palette[m*3+1],palette[m*3+2],r,g,b))
      n|=1<<(7-k);
    }
   }
   /* Store the pattern */
   pp[get_pp_offset(j,i)+5120]=n;
  }
  /* Switch error tables */
  errtemp=err1; err1=err2; err2=errtemp;
  /* Print progress counter */
  if (((i-start_y)&7)==0)
  {
   printf ("%c%c%c%c%3u%%",8,8,8,8,(i-start_y)*100/height);
   fflush (stdout);
  }
 }
 /* Free bitmap buffer allocated by ReadBitmap() */
 free (bmp);
 printf ("%c%c%c%c100%%",8,8,8,8);
 /* Finally, open the output file and write the data */
 printf ("\n Opening %s... ",out);
 fflush (stdout);
 f=fopen (out,"wb");
 if (!f)
 {
  puts ("FAILED");
  return -1;
 }
 printf ("OK\n Writing data... ");
 fflush (stdout);
 if (height<=160 && width<=240)
 {
  /* Write 10k binary */
  if (fwrite(pp+5120,1,5120,f)!=5120)
  {
   puts ("FAILED");
   fclose (f);
   return -1;
  }
  if (fwrite(pp,1,5120,f)!=5120)
  {
   puts ("FAILED");
   fclose (f);
   return -1;
  }
 }
 else
 {
  /* Write 40k or 80k workspace image */
  i=(height<=320)? 40960:81920;
  if (fwrite(pp,1,i,f)!=i)
  {
   puts ("FAILED");
   fclose (f);
   return -1;
  }
 }
 fclose (f);
 puts ("OK");
 return 0;
}

/****************************************************************************/
/* Check program arguments and convert all given files                      */
/****************************************************************************/
int main (int argc,char *argv[])
{
 char *Options[]=
 {
  "palette","dither","scale","debug","centre","interpolate",
  "grey","noise","brightness","weight","gamma", NULL
 };
 int i,j,n,dither=3,centre=0,interpolate=1;
 int palnum=0,scale=0,greyscale=0,noiselevel=50,brightness=50,filternum=0;
 int w1=7,w2=5,w3=3,w4=1;
 int misparm=0;
 char *p;
 float _gamma=1.0;
 static int filters[][11] =
 {
  { 12,0, 1,1,1, 1,4,1, 1,1,1 },          /* Smooth                         */
  { 19,0, 2,2,2, 2,3,2, 2,2,2 },          /* Smooth +                       */
  { 2,0, 0,-1,0, -1,6,-1, 0,-1,0 },       /* Detail                         */
  { 2,0, -1,-1,-1, -1,10,-1, -1,-1,-1 },  /* Edge enhance                   */
  { 1,0, -1,-1,-1, -1,9,-1, -1,-1,-1 },   /* Edge enhance +                 */
  { 1,0, -1,-1,-1, -1,8,-1, -1,-1,-1 },   /* Find edges                     */
  { 1,255, -1,-1,-1, -1,8,-1, -1,-1,-1 }  /* Contour                        */
 };
 static byte palettes[][16*3] =
 {
  /* Coleco ADAM's TMS9918 palette */
  {
     0,  0,  0,    0,  0,  0,   71,183, 59,  124,207,111,
    93, 78,255,  128,114,255,  182, 98, 71,   93,200,237,
   215,107, 72,  251,143,108,  195,205, 65,  211,218,118,
    62,159, 47,  182,100,199,  204,204,204,  255,255,255
  },
  /* Default V9938 palette */
  {
     0,  0,  0,    0,  0,  0,   32,192, 32,   96,224, 96,
    32, 32,224,   64, 96,224,  160, 32, 32,   64,192,224,
   224, 32, 32,  224, 96, 96,  192,192, 32,  192,192,128,
    32,128, 32,  192, 64,160,  160,160,160,  224,224,224
  },
  /* TMS9918 Black & White palette */
  {
     0,  0,  0,    0,  0,  0,  136,136,136,  172,172,172,
   102,102,102,  134,134,134,  120,120,120,  172,172,172,
   136,136,136,  172,172,172,  187,187,187,  205,205,205,
   118,118,118,  135,135,135,  204,204,204,  255,255,255
  },
  /* V9938 Black & White palette */
  {
     0,  0,  0,    0,  0,  0,  144,144,144,  195,195,195,
    60, 60, 60,  115,115,115,   80, 80, 80,  178,178,178,
   102,102,102,  153,153,153,  198,198,198,  211,211,211,
   100,100,100,  129,129,129,  182,182,182,  255,255,255
  }
 };
 printf ("bmp2pp: Windows bitmap to PowerPaint converter\n"
         "Copyright (C) Marcel de Kogel 1997\n"
         "This program supports 2,4,8 and 24 "
         "bits per pixel uncompressed bitmaps\n");
 for (n=1,i=0;n<argc;n++)
 {
  if(*argv[n]!='-') ++i;
  else
  {    
   for(j=0;Options[j];j++)
    if(!strcmp(argv[n]+1,Options[j])) break;
   switch(j)
   {
    case 0:  n++;
             if (n<argc) palnum=atoi(argv[n]);
             else misparm=1;
             break;
    case 1:  n++;
             if (n<argc) dither=atoi(argv[n]);
             else misparm=1;
             break;
    case 2:  n++;
             if (n<argc) scale=atoi(argv[n]);
             else misparm=1;
             break;
    case 3:  debug=1;
             break;
    case 4:  n++;
             if (n<argc) centre=atoi(argv[n]);
             else misparm=1;
             break;
    case 5:  n++;
             if (n<argc) interpolate=atoi(argv[n]);
             else misparm=1;
             break;
    case 6:  greyscale=1;
             break;
    case 7:  n++;
             if (n<argc) noiselevel=atoi(argv[n]);
             else misparm=1;
             break;
    case 8:  n++;
             if (n<argc) brightness=atoi(argv[n]);
             else misparm=1;
             break;
    case 9:  n++;
             if (n<argc)
             {
              w1=atoi(argv[n]);
              p=strchr(argv[n],',');
              if (p)
              {
               w2=atoi(p+1);
               p=strchr(p+1,',');
               if (p)
               {
                w3=atoi(p+1);
                p=strchr(p+1,',');
                if (p) w4=atoi(p+1);
               }
              }
             }
             else misparm=1;
             break;
    case 10: n++;
             if (n<argc) _gamma=atof(argv[n]);
             else misparm=1;
             break;
    case 11: n++;
             if (n<argc) filternum=atoi(argv[n]);
             else misparm=1;
             break;
    default: misparm=1;
             break;
   }
  }
 }
 if (palnum>3 || palnum<0) palnum=0;
 if (filternum>7 || filternum<0) filternum=0;
 if (misparm || !i)
 {
  printf (
   "Usage: bmp2pp [options] <file[,dest]> [file[,dest]...]\n"
   "Available options are:\n"
   "-palette <value>    - Select palette [0]\n"
   "                      0 - original\n"
   "                      1 - V9938\n"
   "                      2 - original/black and white\n"
   "                      3 - V9938/black and white\n"
   "-dither <mode>      - Select dither mode [3]\n"
   "                      0 - No dither\n"
   "                      1 - 1st pass\n"
   "                      2 - 2nd pass\n"
   "                      3 - Both passes\n"
   "-weight <1,2,3,4>   - Set F-S dithering error weights [7,5,3,1]\n"
   "-noise <level>      - Select dithering noise level [50]\n"
   "                      0 - Minimum   100 - Maximum\n"
   "-centre <mode>      - Select centring mode [0]\n"
   "                      0 - Do not centre image\n"
   "                      1 - Centre image to 240x160\n"
   "                      2 - Centre image to 480x320\n"
   "                      3 - Centre image to 480x640\n"
   "-- More --");
  fflush (stdin);
  fgetc (stdin);
  printf ("\n"
   "-scale <mode>       - Select scaling mode [0]\n"
   "                      0 - Do not scale bitmap\n"
   "                      1 - Scale bitmap to 240x160\n"
   "                      2 - Scale bitmap to 480x320\n"
   "                      3 - Scale bitmap to 480x640\n"
   "-interpolate <mode> - Select interpolation mode used for scaling [1]\n"
   "                      0 - Do not use interpolation\n"
   "                      1 - Use linear interpolation\n"
   "-gamma <value>      - Select gamma adjustment [1.0]\n"
   "-brightness <value> - Select image brightness [50]\n"
   "                      0 - Darkest   100 - Brightest\n"
   "-grey               - Convert image to greyscale\n"
   "-filter <value>     - Select filter to apply to image data [0]\n"
   "                      0 - No filter\n"
   "                      1 - Smooth\n"
   "                      2 - Smooth +\n"
   "                      3 - Detail\n"
   "                      4 - Edge enhance\n"
   "                      5 - Edge enhance +\n"
   "                      6 - Find edges\n"
   "                      7 - Contour\n"
   "-debug              - Write pre-processing results to GAMMA.bmp,\n"
   "                      BRIGHT.bmp,GREY.bmp,SCALE.bmp and FILTER.bmp\n");
  return 1;
 }
 /* Initialise colour difference lookup tables */
 for (i=0;i<256;++i)
 {
  bdif[i]=i*i*11;
  gdif[i]=i*i*59;
  rdif[i]=i*i*30;
 }
 /* Check command line arguments for files to convert */
 for (n=1;n<argc;n++)
 {
  if(*argv[n]!='-')
   bmp2pp (argv[n],palettes[palnum],dither,w1,w2,w3,w4,noiselevel,
           scale,interpolate,centre,_gamma,brightness,greyscale,
           (filternum)? filters[filternum-1]:NULL);
  else
  {    
   for(j=0;Options[j];j++)
    if(!strcmp(argv[n]+1,Options[j])) break;
   switch(j)
   {
    case 0: case 1: case 2: case 4: case 5: case 7: case 8: case 9: case 10:
    case 11:
     n++;
    default:
     break;
   }
  }
 }
 return 0;
}
